/*************************************************/
/* Examples Program For "ET-dsPIC33WEB-V1" Board */
/* Hardware    : ET-dsPIC33WEB-V1.0              */
/* Target MCU  : dsPIC33FJ128GP708               */
/*       	   : X-TAL : 8.00 MHz                */
/* Config Fosc : XTAL = 8MHz (XT+PLL)            */
/*             : PLLPRE[4..0] = 00000 (N1=2)     */
/*             : VCO Input = 8MHz / 2 = 4MHz     */
/*             : PLLDIV[8..0] = 0x26(M=40)       */
/*             : VCO Output = 4 x 40MHz = 160MHz */
/*             : PLLPOST[1:0] = 0:0 (N2=2)       */
/*             : Fosc = 160MHz / 2 = 80MHz       */
/*             : Fcy = Fosc/2 = 80MHz /2 = 40MHz */
/* Compiler    : MPLAB + C30 V3.01  		     */
/* Write By    : Eakachai Makarn(ETT CO.,LTD.)   */
/* Last Update : 16/August/2007                  */
/* Function    : Example Use I2C Connect DS1307  */
/*             : Display RTC:Clock:Time on UART1 */	
/*************************************************/
/* Used ET-MINI I/O Ds1307 Module Interface      */
/* SCL1 = RG2(I2C-SCL)                           */
/* SDA1 = RG3(I2C-SDA)                           */
/*************************************************/

#include "p33FJ128GP708.h"                             				// dsPIC33FJ128GP708 MPU Register
#include "stdio.h"													// Used "sprintf" Function

/* Setup Configuration For ET-dsPIC33WEB-V1 */
_FOSCSEL(FNOSC_PRIPLL & 											// Select Primary OSC = XT + PLL
         IESO_OFF )													// Startup With User Select OSC
_FOSC(FCKSM_CSDCMD &        										// Clock Switch Disabled	
      OSCIOFNC_OFF &												// OSC2 Pin = Clock O/P
      POSCMD_XT)													// Primary OSC = XT 
_FWDT(FWDTEN_OFF &           										// Disable Watchdog Timer 
      WINDIS_OFF &          										// Disabled Windowed WDT
      WDTPRE_PR128 &        										// Watchdog prescaler=1:128
      WDTPOST_PS32768)     											// Watchdog postscaler=1:32,768
_FPOR(FPWRT_PWR128)       											// Power-on Reset Value = 128ms
_CONFIG3(JTAGEN_OFF);												// Disable JTAG Interface
/* End of Configuration For ET-dsPIC33WEB-V1 */

char uart1_buf[40]; 												// "sprintf" Buffer

/* pototype  section */
unsigned char ReadDS1307(unsigned char Read_Addr);					// Read RTC:DS1307 Time
void WriteDS1307(unsigned char Write_Addr,							// Write RTC:DS1307 Time
                 unsigned char Set_Time); 
void putchar1(char ch);  											// Put Char To UART-1
void print_uart1(void);												// Print String to UART1 
void delay(unsigned long int);										// Delay Time Function

int main(void)
{  
  unsigned char Hour,Minute,Second; 								// RTC:DS1307 Buffer Data
  unsigned char Last_Second,Time;

  /* Start of PLL Config Fcy = 40MIPS */
  CLKDIV &= 0xFFE0;													// PLLPRE[4..0] = 00000 (N1=2)
  CLKDIV &= 0xFF3F;													// PLLPOST[1:0] = 00 (N2=2)
  PLLFBD  = 0x26;													// PLLDIV[8..0] = 000100110(M=40)
  while(!OSCCONbits.LOCK);											// Wait PLL to Lock
  OSCCONbits.CLKLOCK = 1;											// Disable PLL Modify
  /* End of PLL Config Fly = 40MIPS */

  /* Start of Confog UART1 */	
  U1MODEbits.UARTEN   = 1;											// Enable RXD UART
  U1MODEbits.USIDL    = 0;											// UART Continue in IDLE
  U1MODEbits.IREN     = 0;											// IRDA Disable
  U1MODEbits.RTSMD    = 0;											// RTS = Flow Control		
  U1MODEbits.UEN1     = 0;											// Enable RX,TX Pin
  U1MODEbits.UEN0     = 0;											
  U1MODEbits.WAKE     = 0;											// Disable Wakeup
  U1MODEbits.LPBACK   = 0;											// Disable Loop Back
  U1MODEbits.ABAUD    = 0;											// Disable Auto Baudrate
  U1MODEbits.URXINV   = 0;											// RXD Pin in IDLE = "1"
  U1MODEbits.BRGH     = 0;											// Baudrate = 16 Cycle Clock
  U1MODEbits.PDSEL1   = 0;											// 8 Bit Data,No Parity
  U1MODEbits.PDSEL0   = 0;											
  U1MODEbits.STSEL    = 0;											// 1 Stop Bit
  U1STAbits.UTXISEL1  = 0;											// Interrupt By TX 1-Char
  U1STAbits.UTXISEL0  = 0;										
  U1STAbits.UTXINV	  = 0;											// IRDA Encode in IDLE = 0
  U1STAbits.UTXBRK	  = 0;											// Disable Sync Break
  U1STAbits.UTXEN     = 1;  										// Enable TXD UART 
  U1STAbits.URXISEL1  = 0;											// Interrupt BY RX 1-Char
  U1STAbits.URXISEL0  = 0;										
  U1STAbits.ADDEN	  = 0;											// Disable Address Detect

  // UART[1] Baudrate 
  // ET-dsPIC33WEB-V1 Hardware Board
  // Fcy = 40MHz
  // UxBRG = [Fcy/(16xBaud)]-1
  //       = [ 40MHz / (16 x 9600) ] - 1
  //       = 259 
  U1BRG  = 259;														// UART1 Baudrate 9600/40MHz

  // UART[1] Interrupt Control 
  IEC0bits.U1RXIE     = 0;											// Disable RX Interrupt
  IEC0bits.U1TXIE     = 0;											// Disable TX Interrupt
  /* End of Config UART1 */

  /* Start of Config I2C */
  I2C1CONbits.I2CEN = 0;											// Disable Before Change Config
  I2C1CONbits.I2CSIDL = 1;											// Disable in Idle Mode
  I2C1CONbits.SCLREL = 0;											// I2C Clock Hold
  I2C1CONbits.IPMIEN = 0;   										// Disable I2C IPMI Mode Control
  I2C1CONbits.A10M = 0;  											// I2C Device Address = 7 Bit
  I2C1CONbits.DISSLW = 1;  											// Disable I2C Slew Rate Control
  I2C1CONbits.SMEN = 0;  											// Disable I2C SMBUS Mode
  I2C1CONbits.GCEN = 0; 											// Disable I2C General Call(Slave) 
  I2C1CONbits.STREN = 0;											// Disable SCL Clock Stretch
  I2C1CONbits.ACKDT = 0;											// ACK Cycle = ACK
  I2C1CONbits.ACKEN = 0;											// Disable I2C Acknowledge
  I2C1CONbits.RCEN = 0;												// Disable I2C Receive
  I2C1CONbits.PEN = 0;												// Disable I2C Stop	
  I2C1CONbits.RSEN = 0;												// Disable I2C Restart
  I2C1CONbits.SEN = 0;												// Disable I2C Start
 
  // Setup I2C Baudrate = 100KHz 
  // ET-dsPIC33WEB-V1 Hardware Board
  // Fcy = 40MHz
  // I2CBRG = [(Fcy/Fscl)-(Fcy/1000000)]-1
  //        = [(40 MHz / 100KHz) - (40 MHz / 1000000)] - 1
  //        = [(400)-(40)]-1
  //        = 360 - 1
  //        = 359
  I2C1BRG = 359;													// I2C Baudrate(100 KHz)

  // I2C[1] Interrupt Control 
  IEC1bits.MI2C1IE = 0;												// Disable I2C1 Master Interrupt
  IEC1bits.SI2C1IE = 0;												// Disable I2C1 Slave Interrupt

  I2C1CONbits.I2CEN = 1;											// Enable I2C1 Function
  /* End of Config I2C */

  // Power-On Delay
  delay(1000000);													// Power-On Delay

  // UART[1] Print String //
  sprintf(uart1_buf,"\n\r\rET-dsPIC33WEB-V1.0 : I2C1 + UART1\n\r"); // Print Message String 
  print_uart1();
  sprintf(uart1_buf,"Demo I2C1 Interface RTC-DS1307\n\r"); 			// Print Message String 
  print_uart1();

  //*************************************//
  // Start Setup RTC : DS1307 = 00:00:00 //
  //*************************************//
  WriteDS1307(0x02,0x00);											// Setup Hour = 00	
  WriteDS1307(0x01,0x00);											// Setup Minute = 00
  WriteDS1307(0x00,0x00);											// Setup Second = 00
  Last_Second = 59;													// Default Second

  sprintf(uart1_buf,"Start Setup RTC-Time = 00:00:00\n\r\n\r"); 	// Print Message String 
  print_uart1();

  //********************************//
  // Start Read RTC Clock & Display //
  //********************************//
  while(1)
  { 
    do 																// Repeat Get Second until Second Change
	{
	  Hour   = ReadDS1307(0x02);									// Read Hour (00-23)
	  Minute = ReadDS1307(0x01);									// Read Minute (00-59)
      Second = ReadDS1307(0x00);									// Read Second (00-59)
    }
    while(Last_Second == Second);									// Repeat if Second Not Change
    Last_Second = Second;											// Update Current Second
    
    //************************************//
    // Display Clock = Hour:Minute:Second //
    //************************************//
    sprintf(uart1_buf,"\rDS1307 Clock = ");							// Print Message String 
    print_uart1();
    Time = Hour>>4;													// MSB Hour
    Time &= 0x03;
    Time |= 0x30;
    putchar1(Time);
    Time = Hour;													// LSB Hour
    Time &= 0x0F;
    Time |= 0x30;
    putchar1(Time);
	putchar1(':');	
    Time = Minute>>4;												// MSB Minute
    Time &= 0x07;
    Time |= 0x30;
    putchar1(Time);
    Time = Minute;													// LSB Minute
    Time &= 0x0F;
    Time |= 0x30;
    putchar1(Time);
    putchar1(':');
    Time = Second>>4;												// MSB Second
    Time &= 0x07;
    Time |= 0x30;
    putchar1(Time);
    Time = Second;													// LSB Second
    Time &= 0x0F;
    Time |= 0x30;
    putchar1(Time);
  }
}

/***********************/
/* Delay Time Function */
/*    1-4294967296     */
/***********************/
void delay(unsigned long int count1)
{
  while(count1 > 0) {count1--;}										// Loop Decrease Counter	
}

/************************/
/* Read RTC:DS1307 Time */
/************************/
unsigned char ReadDS1307(unsigned char Read_Addr)					// Read RTC:DS1307 Time
{
  unsigned char Get_Time;											// Time Buffer
	
  I2C1CONbits.SEN = 1;												// Send Start Condition
  while(I2C1CONbits.SEN);											// Wait Start Complete
    
  // Write DS1307 ID Code = 1101000+W
  I2C1TRN = 0xD0;													// Write DS1307 ID Code,Write
  while(I2C1STATbits.TBF);											// Wait Write ID Code Complete

  while(I2C1STATbits.ACKSTAT);										// Wait Acknowledge Complete
  while(I2C1CONbits.SEN || 											// Wait All I2C Status Clear (Wait Idle)
        I2C1CONbits.PEN || 
        I2C1CONbits.RCEN || 
        I2C1CONbits.ACKEN || 
        I2C1STATbits.TRSTAT);

  // Write Address of RTC:ds1307 For Read
  I2C1TRN = Read_Addr;												// Write RTC Address 
  while(I2C1STATbits.TBF);											// Wait Write Address Complete

  while(I2C1STATbits.ACKSTAT);										// Wait Acknowledge Complete
  while(I2C1CONbits.SEN || 											// Wait All I2C Status Clear (Wait Idle)
        I2C1CONbits.PEN || 
        I2C1CONbits.RCEN || 
        I2C1CONbits.ACKEN || 
        I2C1STATbits.TRSTAT);
 
  // Restart For Read DS1307 Data
  I2C1CONbits.RSEN = 1;												// Send Re-Start Condition
  while(I2C1CONbits.RSEN);											// Wait Re-Start Complete
     
  // Write DS1307 ID Code = 1101000+R 
  I2C1TRN = 0xD1;													// Write DS1307 ID Code,Read
  while(I2C1STATbits.TBF);											// Wait Write ID Code Complete

  while(I2C1STATbits.ACKSTAT);										// Wait Acknowledge Complete
  while(I2C1CONbits.SEN || 											// Wait All I2C Status Clear (Wait Idle)
        I2C1CONbits.PEN || 
        I2C1CONbits.RCEN || 
        I2C1CONbits.ACKEN || 
        I2C1STATbits.TRSTAT);

  I2C1CONbits.RCEN = 1;												// Read Time From RTC				
  while(I2C1CONbits.RCEN);
  I2C1STATbits.I2COV = 0;
  Get_Time = I2C1RCV;												// Get I2C Result
  while(I2C1STATbits.RBF);											// Wait Read Data Complete

  while(I2C1STATbits.ACKSTAT);										// Wait Acknowledge Complete
  while(I2C1CONbits.SEN || 											// Wait All I2C Status Clear (Wait Idle)
        I2C1CONbits.PEN || 
        I2C1CONbits.RCEN || 
        I2C1CONbits.ACKEN || 
        I2C1STATbits.TRSTAT);
        
  I2C1CONbits.PEN = 1;												// Send Stop Condition
  while(I2C1CONbits.PEN);											// Wait Stop Complete    

  return Get_Time;													// Return Time Result
}

/*************************/
/* Write RTC:DS1307 Time */
/*************************/
void WriteDS1307(unsigned char Write_Addr,							// Write RTC:DS1307 Time
                 unsigned char Set_Time)						
{
  I2C1CONbits.SEN = 1;												// Send Start Condition
  while(I2C1CONbits.SEN);											// Wait Start Complete

  // Write DS1307 ID Code = 1101000+W  
  I2C1TRN = 0xD0;													// Write DS1307 ID Code (1101000+W)
  while(I2C1STATbits.TBF);											// Wait Write ID Code Complete

  while(I2C1STATbits.ACKSTAT);										// Wait Acknowledge Complete
  while(I2C1CONbits.SEN || 											// Wait All I2C Status Clear (Wait Idle)
        I2C1CONbits.PEN || 
        I2C1CONbits.RCEN || 
        I2C1CONbits.ACKEN || 
        I2C1STATbits.TRSTAT);  
         
  // Write RTC Address
  I2C1TRN = Write_Addr;												// Write RTC Address
  while(I2C1STATbits.TBF);											// Wait Write Address Complete

  while(I2C1STATbits.ACKSTAT);										// Wait Acknowledge Complete
  while(I2C1CONbits.SEN || 											// Wait All I2C Status Clear (Wait Idle)
        I2C1CONbits.PEN || 
        I2C1CONbits.RCEN || 
        I2C1CONbits.ACKEN || 
        I2C1STATbits.TRSTAT);

  // Write Time to RTC:DS1307   
  I2C1TRN = Set_Time;												// Write Time to RTC:DS1307
  while(I2C1STATbits.TBF);											// Wait Write Time Complete

  while(I2C1STATbits.ACKSTAT);										// Wait Acknowledge Complete
  while(I2C1CONbits.SEN || 											// Wait All I2C Status Clear (Wait Idle)
        I2C1CONbits.PEN || 
        I2C1CONbits.RCEN || 
        I2C1CONbits.ACKEN || 
        I2C1STATbits.TRSTAT);
  
  I2C1CONbits.PEN = 1;												// Send Stop Condition
  while(I2C1CONbits.PEN);											// Wait Stop Complete
}

/****************************/
/* Write Character To UART1 */
/****************************/
void putchar1 (char c)  
{
  if (c == '\n')  													// If Line Feed(LF)
  {
    U1TXREG = 0x0D;                        							// Auto Add CR(LF+CR)
    while (!U1STAbits.TRMT);										// Wait Send Complete
    U1STAbits.TRMT = 0;												// Reset Status
  }
  U1TXREG = c;														// Send This Char
  while (!U1STAbits.TRMT);											// Wait Send Complete
  U1STAbits.TRMT = 0;												// Reset Status
}

/*************************/
/* Print String to UART1 */
/*************************/
void print_uart1(void)
{
  char *p;															// Pointer Buffer
  p = uart1_buf;													// UART1 Buffer
 
  do 																// Get char & Print Until null
  {
    putchar1(*p); 													// Write char to UART1
    p++;															// Next char
  }
  while(*p != '\0');												// End of ASCII (null)
 
  return;
}

